/**
 * API route for fetching all events
 */

import { NextRequest, NextResponse } from 'next/server';
import { getAllEvents, getCountries, getEventTypes } from '@/lib/events-store';
import { EventFilters, EventType } from '@/types/event';
import { isWithinInterval, parseISO, startOfWeek, endOfWeek, startOfMonth, endOfMonth, addMonths } from 'date-fns';

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    
    // Parse filter parameters
    const filters: EventFilters = {
      search: searchParams.get('search') || '',
      eventType: (searchParams.get('eventType') as EventType | 'all') || 'all',
      country: searchParams.get('country') || 'all',
      dateRange: (searchParams.get('dateRange') as EventFilters['dateRange']) || 'all',
      ticketAvailable: searchParams.get('ticketAvailable') === 'true' 
        ? true 
        : searchParams.get('ticketAvailable') === 'false' 
          ? false 
          : null,
    };

    let events = getAllEvents();

    // Apply search filter
    if (filters.search) {
      const searchLower = filters.search.toLowerCase();
      events = events.filter(e => 
        e.name.toLowerCase().includes(searchLower) ||
        e.description.toLowerCase().includes(searchLower) ||
        e.venue.toLowerCase().includes(searchLower) ||
        e.city.toLowerCase().includes(searchLower) ||
        e.country.toLowerCase().includes(searchLower) ||
        e.attendeeProfiles.toLowerCase().includes(searchLower)
      );
    }

    // Apply event type filter
    if (filters.eventType && filters.eventType !== 'all') {
      events = events.filter(e => e.eventType === filters.eventType);
    }

    // Apply country filter
    if (filters.country && filters.country !== 'all') {
      events = events.filter(e => e.country === filters.country);
    }

    // Apply date range filter
    if (filters.dateRange && filters.dateRange !== 'all') {
      const today = new Date();
      let startDate: Date;
      let endDate: Date;

      switch (filters.dateRange) {
        case 'this-week':
          startDate = startOfWeek(today, { weekStartsOn: 0 });
          endDate = endOfWeek(today, { weekStartsOn: 0 });
          break;
        case 'this-month':
          startDate = startOfMonth(today);
          endDate = endOfMonth(today);
          break;
        case 'next-month':
          const nextMonth = addMonths(today, 1);
          startDate = startOfMonth(nextMonth);
          endDate = endOfMonth(nextMonth);
          break;
        default:
          startDate = new Date(0);
          endDate = new Date(8640000000000000);
      }

      events = events.filter(e => {
        const eventDate = parseISO(e.date);
        return isWithinInterval(eventDate, { start: startDate, end: endDate });
      });
    }

    // Apply ticket availability filter
    if (filters.ticketAvailable !== null) {
      events = events.filter(e => e.ticketAvailable === filters.ticketAvailable);
    }

    // Sort by date (upcoming first)
    events.sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());

    // Get filter options
    const countries = getCountries();
    const eventTypes = getEventTypes();

    return NextResponse.json({
      events,
      total: events.length,
      filters: {
        countries,
        eventTypes,
      },
    });
  } catch (error) {
    console.error('Error fetching events:', error);
    return NextResponse.json(
      { error: 'Failed to fetch events' },
      { status: 500 }
    );
  }
}

